//
//  AsReaderDevice.h
//  AsReaderDockSDK
//
//  Created by Mikiko on 11/7/24.
//  Copyright © 2024 SPS INC. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <AsReaderBLESDK/AsReaderBLESDK.h>
#import "AsReaderRFIDProtocol.h"
#import "AsReaderBarcodeProtocol.h"
#import "AsReaderNFCProtocol.h"
#import "AsReaderLfProtocol.h"
#import "AsReaderDeviceProtocol.h"
#import "AsReaderInfoDefine.h"

typedef NS_ENUM(NSInteger, SaveType){
    SaveType_Permanet,
    SaveType_Temporary
};

/**
 *  @brief      This class is facilitates communication with various AsReader devices, including barcode scanners, RFID readers, NFC devices, and BLE (Bluetooth Low Energy) devices.
 */
@interface AsReaderDevice : NSObject

/**
 *  @brief  RFID event delegate
 *  @details Only RFID (See <AsReaderRFIDDeviceDelegate>)
 */
@property (nonatomic, weak, setter = setDelegateRFID:) id<AsReaderRFIDDeviceDelegate> delegateRFID;

/**
 *  @brief  RFID event delegate
 *  @details Only RFID (See <AsreaderBarcodeDeviceProtocol>)
 */
@property (nonatomic, weak, setter = setDelegateBarcode:, 
	getter = getDelegateBarcode) id<AsreaderBarcodeDeviceDelegate> delegateBarcode;

/**
 *  @brief  RFID event delegate
 *  @details Only RFID (See <AsReaderNFCDeviceDelegate>)
 */
@property (nonatomic, weak, setter = setDelegateNFC:) id<AsReaderNFCDeviceDelegate> delegateNFC;

/**
 *  @brief  LF event delegate
 *  @details Only LF (See <AsReaderLfDeviceDelegate>)
 */
@property (nonatomic, weak, setter = setDelegateLf:) id<AsReaderLfDeviceDelegate> delegateLf;

/**
 *  @brief  RFID event delegate
 *  @details Device (See <AsReaderDeviceDelegate>)
 */
@property (nonatomic, weak, setter = setDelegateDevice:) id<AsReaderDeviceDelegate> delegateDevice;

@property (nonatomic, assign) ConnectionType connectionType;

//*******************************************************************************//
//  Common API (RFID, NFC, Barcode)
//  This function can use all device
//*******************************************************************************//

#pragma mark - class methods
// 
+ (NSString *)getSDKVersion;

+ (void)showPrintNSLog:(BOOL)isShow;

// DO NOT USE THIS. deprecated. please use the instance method instead.
+ (void)setTriggerModeDefault:(BOOL)isDefault __deprecated_msg("use the instance method instead.");

// DO NOT USE THIS. deprecated. moved to AsReaderRFIDDevice
+ (void)setReadRSSIMode:(BOOL)isReadRSSIMode __deprecated_msg("use the instance method instead.");

#pragma mark - instance methods

// deprecated. moved to AsReaderRFIDDevice
//- (BOOL)getReaderInformation;

- (BOOL)sendToReader:(NSData*)data;



/**
 *  @brief      Send the "Get Reader Information" command to the reader
 *  @details    Get basic information from the reader.
 *  @param      infoType : model(0x00), RFID Version(0x01), Manufacturer(0x02), Frequency(0x03), Tag Type(0x04)
 *  @return     Returns NO if selected mode of reader is not RFID, YES otherwise
 */
- (BOOL)getReaderInfo:(int)infoType;

/**
 *  @brief      Get current battery level
 *  @return     Returns battery level
 */
- (int)getCurrentBattery;

// DO NOT USE THIS. deprecated.
- (BOOL)setBeep:(BOOL)beepOn setVibration:(BOOL)vibrationOn setIllumination:(BOOL)illuminationOn setLED:(BOOL)led __deprecated;;

/**
 *  @brief      Send the "setting" command to the reader
 *  @details    Setting when reading a tag. Can set beep, vibration, illumination, led.
 *  @param      isBeep : On(1), Off(0)
 *  @param      isVibration : On(1), Off(0)
 *  @param      isbatteryGaugeLed : On(1), Off(0)
 *  @param      isBarcodeAimer : On(1), Off(0)
 *  @return     YES
 */
- (BOOL)setBeep:(BOOL)isBeep 
      vibration:(BOOL)isVibration
batteryGaugeLed:(BOOL)isbatteryGaugeLed
   barcodeAimer:(BOOL)isBarcodeAimer;

// DO NOT USE THIS. deprecated.
- (int)setSettingBeep:(BOOL)isBeep vibration:(BOOL)isVib led:(BOOL)isLed aimer:(BOOL)isAimer powerOnBeep:(BOOL)powerOnBeep __deprecated;

/**
 *  @brief      beep and other options.
 *  @details    Set  beep, vibration, led, aimer, and  powerOnBeep can be set at the same time.
 *  @param      isBeep : On(YES), Off(NO)
 *  @param      isVibration : On(YES), Off(NO)
 *  @param      isbatteryGaugeLed On(YES), Off(NO)
 *  @param      isBarcodeAimer : On(YES), Off(NO)
 *  @param      isBarcodePowerOnBeep :On(YES), Off(NO)
 *  @return     YES
 */
- (BOOL)setBeep:(BOOL)isBeep vibration:(BOOL)isVibration batteryGaugeLed:(BOOL)isbatteryGaugeLed barcodeAimer:(BOOL)isBarcodeAimer barcodePowerOnBeep:(BOOL)isBarcodePowerOnBeep;

// DO NOT USE THIS. deprecated.
- (int)setReaderPower:(BOOL)isOn beep:(BOOL)isBeep vibration:(BOOL)isVib led:(BOOL)isLed illumination:(BOOL)isIllu mode:(int)nDeviceType __deprecated;
// DO NOT USE THIS. deprecated.
- (int)setReaderPower:(BOOL)isOn beep:(BOOL)isBeep vibration:(BOOL)isVib led:(BOOL)isLed illumination:(BOOL)isIllu connectedBeep:(BOOL)isConnectedBeep mode:(int)nDeviceType __deprecated;

/**
 *  @brief      Set reader power on/off with beep and other options.
 *  @details    Set reader power on/off with options. When set power on, beep, vibration, illumination, led, and PowerOnBeep can be set at the same time.
 *  @param      isOn : On(YES), Off(NO)
 *  @param      isBeep : On(YES), Off(NO)
 *  @param      isVib : On(YES), Off(NO)
 *  @param      isbatteryGaugeLed : On(YES), Off(NO)
 *  @param      isBarcodeAimer : On(YES), Off(NO)
 *  @param      isBarcodePowerOnBeep : On(YES), Off(NO)
 *  @param      nDeviceType : type of reader
 *  @return     Returns 0 if a command sent failed, 1 if a command is added to the queue.
 */
- (int)setReaderPower:(BOOL)isOn 
	beep:(BOOL)isBeep 
	vibration:(BOOL)isVib 
	batteryGaugeLed:(BOOL)isbatteryGaugeLed 
	barcodeAimer:(BOOL)isBarcodeAimer 
	barcodePowerOnBeep:(BOOL)isBarcodePowerOnBeep 
	mode:(int)nDeviceType;

/**
 *  @brief      Send the "Set Stop Condition" command to the reader.
 *  @details    Set the stop point of start-auto-read.\n Should only be used on RFID type.
 *  @param      mtnu : Maximum number of tag to read
 *  @param      mtime : Maximum elapsed time to tagging(sec)
 *  @param      repeatCycle : How many times reader perform inventory round
 */
- (void)setTagCount:(int)mtnu
        setScanTime:(int)mtime
           setCycle:(int)repeatCycle;


- (BOOL) setChargingControl:(BOOL)isOn;


- (void) setDelayDisconnectOnBackground:(NSTimeInterval)time;

- (NSTimeInterval) getDelayDisconnectOnBackground;

/**
 *  @brief     return connection status.
 *  @return    plugged (1) , unplugged (0)
 */
- (BOOL)isOpened;


/**
 *  @brief      Send the "Set Stop Condition" command to the reader.
 *  @details    Set the stop point of start-auto-read
 *  @param      maxTags : Maximum number of tags to read
 *  @param      maxTime : Maximum elapsed time to read tags(sec)
 *  @param      repeatCycle : How many times the reader performs an inventory round
 *  @return     If the connected reader is RFID jacket type, return NO, otherwise, return YES.
 */
- (BOOL)setStopTagNum:(int)maxTags
              maxTime:(int)maxTime
          repeatCycle:(int)repeatCycle;


/**
 *  @brief      Send the "Get Stop Condition" command to the reader.
 *  @details    Get the stop point of start-auto-read
 *  @return     If the connected reader is RFID jacket type, return NO, otherwise, return YES.
 */
- (BOOL)getStopCondition;

- (BOOL)setTriggerMode:(BOOL)isDefault;

// DO NOT USE THIS. deprecated. please use the 'setTriggerMode' instead.
- (void)setTriggerModeDefault:(BOOL)isDefault __deprecated_msg("use the 'setTriggerMode' instead.");

- (BOOL)reset;

- (BOOL)getAutoLaunch;

- (BOOL)setAutoLaunch:(BOOL)enable bundleId:(NSString*) bundleId;

- (BOOL)setSleepBeep: (BOOL)isOn;

- (BOOL)setSleepTime: (int)time;

- (BOOL)getSleepTime;


#pragma mark - BLE

/**
 *  @brief      BLE device discovery
 *  @details    BLE device discovery
 *  @return     YES : Success NO : If it is already connected, disconnect it..
 */
-(BOOL)startBleScan;

/**
 *  @brief      Stop searching for BLE devices
 *  @details    Stop searching for BLE devices
 */
- (void)stopBleScan;


/**
 *  @brief      Disconnect the connected BLE device. .
 *  @details    Disconnect the connected BLE device. .
 */
- (void)disConnectBLE;

/**
 *  @brief      Attempt to connect to a BLE device
 *  @details    Attempt to connect the selected peripheral device
 */
- (void)connectBLE:(id)peripheral;

- (int)getSleepTimeForBLEDevice;


- (int)setSleepTimeForBLEDevice:(int)min type:(SaveType)type;
@end
