//
//  AsReaderGUNManager.swift
//  AsReaderGUN
//
//  Created by mac on 2020/5/28.
//  Copyright © 2020 Asterisk. All rights reserved.
//

import UIKit

let deviceModel = "com.asreader.gun"
let asreaderGUNDisconnected = "AsReaderGUNDisconnected"
let asreaderGUNConnected = "AsReaderGUNConnected"
let asreaderGUNAlert = "Alert"

@objc protocol AsReaderGUNManagerDelegate: NSObjectProtocol {
    @objc optional func whenAsReaderGUNConnected(_ isConnected: Bool)
    @objc optional func asReaderGUNUpdateDeviceState(_ error: ResultCode)
    @objc optional func asReaderGUNReadTag(_ tag: String?, rssi: Float, phase: Float, frequency: Float)
    @objc optional func asReaderGUNChangedActionState(_ action: CommandType, resultCode: Int)
    @objc optional func asReaderGUNAccessResult(_ error: ResultCode, actionState action: CommandType, epc: String?, data: String?, rssi: Float, phase: Float, frequency: Float)
    @objc optional func asReaderGUNDetect(_ barcodeType: BarcodeType, codeId: String!, barcodeData: Data!)
    @objc optional func asReaderGUNReadTag(withTid error: ResultCode, actionState action: CommandType, epc: String!, tid data: String!, rssi: Float, phase: Float, frequency: Float)
    @objc optional func asReaderGUNOn(asReaderTriggerKeyEvent status: Bool) -> Bool
    @objc optional func asReaderGUNOn(asReaderLeftModeKeyEvent status: Bool) -> Bool
    @objc optional func asReaderGUNOn(asReaderRightModeKeyEvent status: Bool) -> Bool
    @objc optional func asReaderGUNCommandComplete(_ command: CommandType)
    @objc optional func asReaderGUNOnModuleOverHeated()
    @objc optional func asReaderGUNOnResetReader()
    @objc optional func asReaderGUNReadTagWithRFM(_ error: ResultCode, action: CommandType, epc: String!, rssi: Float, phase: Float, frequency: Float, tid: String!, tempCode: String!, calibration: String!, sensorCode: String!)
}

class AsReaderGUNManager: NSObject, AsReaderDelegate {
    weak var asReaderGUNManagerDelegate: AsReaderGUNManagerDelegate?
    var mAsReaderGUN: AsReaderGUN!
    var mReader: AsReader?
    var backupUseMask: SelectFlag!
    var backupSelectMasks: [AsSelectMaskParam] = []
    var systemSetting: SystemSetting!
    static let sharedAsReaderGUNManager = AsReaderGUNManager()
    private var isSDKAlert: Bool = false
    override init() {
        super.init()
        NotificationCenter.default.addObserver(self, selector: #selector(asReaderGUNConnected),
                                               name: NSNotification.Name(rawValue: asreaderGUNConnected),
                                               object: nil)
        NotificationCenter.default.addObserver(self, selector: #selector(asReaderGUNDisconnected),
                                               name: NSNotification.Name(rawValue: asreaderGUNDisconnected),
                                               object: nil)
        NotificationCenter.default.addObserver(self, selector: #selector(SDKAlert),
        name: NSNotification.Name(rawValue: asreaderGUNAlert),
        object: nil)
        systemSetting = SystemSetting.shared()
        self.mAsReaderGUN = AsReaderGUN(deviceModel: deviceModel)
    }
    @objc private func asReaderGUNConnected() {
        DispatchQueue.main.async {
            self.mReader = nil
            self.mReader = AsReader.init(asReaderGUN: self.mAsReaderGUN, delegate: self)
            self.mReader?.setDelegate(self)
        }
    }
    @objc private func asReaderGUNDisconnected() {
        self.mReader = nil
        asReaderGUNManagerDelegate?.whenAsReaderGUNConnected?(false)
    }
    @objc private func SDKAlert(notification:NSNotification) {
        DispatchQueue.main.async {
            if self.isSDKAlert == false  {
                guard let dic: Dictionary = notification.userInfo else { return }
                guard let dataString: String = dic["DATA"] as? String else { return }
                let alertView = UIAlertView.init(title: "Alert", message: dataString, delegate: nil, cancelButtonTitle: "OK")
                alertView.show()
            }
        }
    }
    func readerInitialized(_ reader: AsReader?) {
        asReaderGUNManagerDelegate?.whenAsReaderGUNConnected?(true)
    }
    func updateDeviceState(_ error: ResultCode) {
        asReaderGUNManagerDelegate?.asReaderGUNUpdateDeviceState?(error)
    }
    func readTag(_ tag: String!, rssi: Float, phase: Float, frequency: Float) {
        asReaderGUNManagerDelegate?.asReaderGUNReadTag?(tag, rssi: rssi, phase: phase, frequency: frequency)
    }
    func changedActionState(_ action: CommandType, resultCode: Int) {
        asReaderGUNManagerDelegate?.asReaderGUNChangedActionState?(action, resultCode: resultCode)
    }
    func accessResult(_ error: ResultCode, actionState action: CommandType, epc: String?, data: String?, rssi: Float, phase: Float, frequency: Float) {
        asReaderGUNManagerDelegate?.asReaderGUNAccessResult?(error, actionState: action, epc: epc, data: data, rssi: rssi, phase: phase, frequency: frequency)
    }
    func detect(_ barcodeType: BarcodeType, codeId: String!, barcodeData: Data!) {
        asReaderGUNManagerDelegate?.asReaderGUNDetect?(barcodeType, codeId: codeId, barcodeData: barcodeData)
    }
    func readTag(withTid error: ResultCode, actionState action: CommandType, epc: String!, tid data: String!, rssi: Float, phase: Float, frequency: Float) {
        asReaderGUNManagerDelegate?.asReaderGUNReadTag?(withTid: error, actionState: action, epc: epc, tid: data, rssi: rssi, phase: phase, frequency: frequency)
    }
    func readTag(withRFM error: ResultCode, actionState action: CommandType, epc: String!, rssi: Float, phase: Float, frequency: Float, tid: String!, temCode tempCode: String!, calibaration calibration: String!, sensorCode: String!) {
        asReaderGUNManagerDelegate?.asReaderGUNReadTagWithRFM?(error, action: action, epc: epc, rssi: rssi, phase: phase, frequency: frequency, tid: tid, tempCode: tempCode, calibration: calibration, sensorCode: sensorCode)
    }
    func on(asReaderTriggerKeyEvent status: Bool) -> Bool {
        return (asReaderGUNManagerDelegate?.asReaderGUNOn?(asReaderTriggerKeyEvent: status))!
    }
    func on(asReaderLeftModeKeyEvent status: Bool) -> Bool {
        return (asReaderGUNManagerDelegate?.asReaderGUNOn?(asReaderLeftModeKeyEvent: status))!
    }
    func on(asReaderRightModeKeyEvent status: Bool) -> Bool {
        return (asReaderGUNManagerDelegate?.asReaderGUNOn?(asReaderRightModeKeyEvent: status))!
    }
    func commandComplete(_ command: CommandType) {
        asReaderGUNManagerDelegate?.asReaderGUNCommandComplete?(command)
    }
    func onModuleOverHeated() {
        asReaderGUNManagerDelegate?.asReaderGUNOnModuleOverHeated?()
    }
    func onResetReader() {
        asReaderGUNManagerDelegate?.asReaderGUNOnResetReader?()
    }

    func initMask(_ maskValue: String?, maskType: MaskType) {
        guard let nowMaskValue = maskValue else {
            return
        }
        guard !nowMaskValue.isEmpty else {
            return
        }
        if maskType == MaskType_Selection {
            backupUseMask = mReader?.useSelectionMask
            for index in MIN_SELECTION_MASK..<MAX_SELECTION_MASK {
                let result = mReader?.usedSelectionMask(index) ?? false
                if !result {
                    break
                }
                let asSelectMaskParam: AsSelectMaskParam = (mReader?.getSelectionMask(index))!
                backupSelectMasks.append(asSelectMaskParam)
            }
        }
        let param = AsSelectMaskParam(parameterIndex: 0, target: MaskTarget_SL, action: MaskAction_AB, bank: Bank_EPC, offset: 32, mask: (nowMaskValue as NSString?)?.substring(from: 4), used: true)
        mReader?.setSelectionMask(0, with: param)
        mReader?.useSelectionMask = SelectFlag_SL
        for index in 1..<MAX_SELECTION_MASK {
            mReader?.setMaskUsed(index, used: false)
        }
    }
    func exitMask() {
        if let aBackupUseMask = backupUseMask {
            mReader?.useSelectionMask = aBackupUseMask
        }
        for index in MIN_SELECTION_MASK..<MAX_SELECTION_MASK {
            print("count: \(backupSelectMasks.count)")
            if backupSelectMasks.count < index + 1 {
                mReader?.removeSelectionMask(index)
                break
            }
            let param = backupSelectMasks[Int(index)]
            mReader?.setSelectionMask(index, with: param)
        }
        backupSelectMasks.removeAll()
        //EPC MASK Rollback
        if mReader?.maskTypeValue == Int32(MaskType_EPC.rawValue) {
            rollBackEpcMask()
        }
    }
    func rollBackEpcMask() {
        var count: Int32 = mReader?.epcMaskCount() ?? 0
        NSLog(" EPC MASK Count : %d", count)
        var maskData: [AsSelectMaskEPCParam] = []
        for index in 0..<count {
            let param = mReader?.getEpcMask(index)
            if param != nil {
                maskData.append(param!)
            }
        }
        mReader?.clearEpcMask()
        for mask: AsSelectMaskEPCParam in maskData {
            mReader?.addEpcMask(mask)
        }
        let status: Bool = mReader?.epcMaskMatchMode ?? false
        mReader?.epcMaskMatchMode = status
        count = mReader?.epcMaskCount() ?? 0
        print(" EPC MASK Count : %d", count)
    }
    //epcMaskMatchMode
    func getCurrentVC() -> UIViewController? {
        var result: UIViewController?
        var window: UIWindow? = UIApplication.shared.keyWindow
        if window?.windowLevel != UIWindow.Level.normal {
            let windows = UIApplication.shared.windows
            for tmpWin: UIWindow in windows  where tmpWin.windowLevel == UIWindow.Level.normal {
                window = tmpWin
                break
            }
        }
        let frontView: UIView? = window?.subviews[0]
        let nextResponder = frontView?.next
        if nextResponder is UIViewController {
            result = nextResponder as? UIViewController
        } else {
            result = window?.rootViewController
        }
        return result
    }

    final class func showGlobalProgressHUD(withTitle title: String?) {
        var result: UIViewController?
        var window: UIWindow? = UIApplication.shared.keyWindow
        if window?.windowLevel != UIWindow.Level.normal {
            let windows = UIApplication.shared.windows
            for tmpWin: UIWindow in windows where tmpWin.windowLevel == UIWindow.Level.normal {
                window = tmpWin
                break
            }
        }
        let frontView: UIView? = window?.subviews[0]
        let nextResponder = frontView?.next
        if nextResponder is UIViewController {
            result = nextResponder as? UIViewController
        } else {
            result = window?.rootViewController
        }
        guard let nowResult = result else {
            return
        }
        guard let view = nowResult.view  else {
            return
        }
        guard let nowTitle = title else {
            return
        }
        let hud = MBProgressHUD.showAdded(to: view, animated: true)
        hud.label.text = nowTitle
    }

    final class func dismissGlobalHUD() {
        var result: UIViewController?
        var window: UIWindow? = UIApplication.shared.keyWindow
        if window?.windowLevel != UIWindow.Level.normal {
            let windows = UIApplication.shared.windows
            for tmpWin: UIWindow in windows  where tmpWin.windowLevel == UIWindow.Level.normal {
                window = tmpWin
                break
            }
        }
        let frontView: UIView? = window?.subviews[0]
        let nextResponder = frontView?.next
        if nextResponder is UIViewController {
            result = nextResponder as? UIViewController
        } else {
            result = window?.rootViewController
        }
        guard let nowResult = result else {
            return
        }
        guard let view = nowResult.view  else {
            return
        }
        MBProgressHUD.hide(for: view, animated: true)
    }
    func asReaderGUNSetting() {
        if self.mReader != nil {
            self.mReader?.buzzer = BuzzerState(rawValue: BuzzerState.RawValue(systemSetting.getBuzzer()))
            self.mReader?.vibrator = VibratorState(rawValue: VibratorState.RawValue(systemSetting.getVibrator()))
            self.mReader?.sleepTime = Int32(systemSetting.getSleepTime())
        }
    }
    func rfidSetting() {
        if self.mReader != nil {
            self.mReader?.inventoryTime = Int32(systemSetting.getInventoryTime())
            self.mReader?.idleTime = Int32(systemSetting.getIdleTime())
            self.mReader?.autoOffTime = Int32(systemSetting.getAutoOffTime())
            self.mReader?.qValue = Int32(systemSetting.getQValue())
            self.mReader?.algorithm = AlgorithmType(rawValue: AlgorithmType.RawValue(systemSetting.getAlgorithm()))
            self.mReader?.maxQ = Int32(systemSetting.getMaxQ())
            self.mReader?.minQ = Int32(systemSetting.getMinQ())
            self.mReader?.linkProfileValue = Int32(systemSetting.getLinkProfileValue())
            self.mReader?.defaultLinkProfileValue = Int32(systemSetting.getDefaultLinkProfileValue())
            self.mReader?.operationTime = Int32(systemSetting.getOperationTime())
            self.mReader?.powerGain = Int32(systemSetting.getPowerGain())
            self.mReader?.inventorySession = SessionType(rawValue: SessionType.RawValue(systemSetting.getInventorySession()))
            self.mReader?.sessionFlag = SessionFlag(rawValue: SessionFlag.RawValue(systemSetting.getSessionFlag()))
            self.mReader?.continuousMode = systemSetting.getContinuousMode()
            self.mReader?.rssiMode = systemSetting.getRssiMode()
            let regionName: String = self.mReader?.regionName() ?? ""
            if regionName.hasPrefix("Japan") {
                var lbtItemArr = systemSetting.getLbtItemData()
                if lbtItemArr == nil {
                    if let lbtItems = self.mReader?.getLBT() {
                        if var lbtArr: [LbtItem] = lbtItems as? [LbtItem] {
                            var lbtI: LbtItem = LbtItem.init()
                            for lbtItem in lbtArr {
                                let lbitem = lbtItem
                                lbitem.frequency = self.mReader?.getLBTFrequency(lbitem.mSlot)
                                if lbitem.frequency == "0 HZ" {
                                    lbtI = lbitem
                                }
                            }
                            if let index = lbtArr.firstIndex(of: lbtI) {
                                lbtArr.remove(at: index)
                            }
                            lbtItemArr = lbtArr
                            systemSetting.setLbtItemData(lbtItemArr)
                        }
                    }
                }
                self.mReader?.setLBT(systemSetting.getLbtItemData())
            }
            self.mReader?.maskTypeValue = Int32(systemSetting.getMaskTypeValue())
            switch UInt32(systemSetting.getMaskTypeValue()) {
            case MaskType_NO_MASK.rawValue:
                break
            case MaskType_Selection.rawValue:
                var param: AsSelectMaskParam?
                if let epcMaskData = systemSetting.getEpcMaskData() {
                    let selectionArray: [AsSelectMaskParam] = epcMaskData
                    for index in MIN_SELECTION_MASK..<MAX_SELECTION_MASK {
                        if index < Int(selectionArray.count) {
                            param = selectionArray[Int(index)]
                            self.mReader?.setSelectionMask(index, with: param)
                        } else {
                            self.mReader?.removeSelectionMask(index)
                        }
                    }
                    if selectionArray.isEmpty {
                        self.mReader?.useSelectionMask = SelectFlag(rawValue: SelectFlag.RawValue(systemSetting.getUseSelectionMask()))
                    } else {
                        self.mReader?.clearSelectionMask()
                    }
                  } else {
                    self.mReader?.clearSelectionMask()

                }
            case MaskType_EPC.rawValue:
                self.mReader?.clearEpcMask()
                 if let selectionEpcMaskData = systemSetting.getSelectionEpcMaskData() {
                    for mask in selectionEpcMaskData {
                        self.mReader?.addEpcMask(mask)
                    }
                }
                if systemSetting.getEpcMaskMatchMode() {
                    self.mReader?.epcMaskMatchMode = true
                } else {
                    self.mReader?.epcMaskMatchMode = false
                }
            default:
                break
            }
        }
    }

    func barcodeSetting() {
        if self.mReader != nil {
            self.mReader?.setPropBarcodeMode(true)
            var resultCode = ResultNoError
            if let arr = systemSetting.getBarcodeParamData() {
                for paramName in arr {
                    resultCode = self.mReader?.setBarcodeParam(paramName, type:Temporary) ?? ResultNoError
                    if resultCode != ResultNoError {
                        return
                    }
                }
            }
            self.mReader?.setPropBarcodeMode(false)
        }
    }

}
